/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: XIAO Tong (email: xiaotong@mail.neu.edu.cn) 2018-04-24
*/

#include "CopyBlocksOnSite.h"
#include "CopyBlocksOnSite.cuh"
#include "../XDevice.h"

namespace nts { // namespace nts(NiuTrans.Tensor)

#ifdef USE_CUDA

/*
copy a number of blocks to target positions
NOTE that this version makes more use of the 2d threads in cuda
>> source - data array (head of the blocks) to copy from
>> blockSize - size of block
>> blockNum - number of blocks
>> target - target data array
>> targetBlocks - target positions of the copy
*/
template<int miniBlockSize>
__global__
void KernelCopyBlocks(DTYPE * source, int blockSize, int blockNum, DTYPE * target, int * targetBlocks)
{
    /* entry index in the block */
    int i = (blockDim.x * blockIdx.x + threadIdx.x) * miniBlockSize;

    /* block index */
    int j = blockDim.y * blockIdx.y + threadIdx.y;

    if (j >= blockNum)
        return;

    /* target position */
    int k = targetBlocks[j];

    DTYPE * s = source + blockSize * j;
    DTYPE * t = target + blockSize * k;

    if (i < blockSize) {
        if (miniBlockSize == 4) {
            t[i] = s[i];
            t[i + 1] = s[i + 1];
            t[i + 2] = s[i + 2];
            t[i + 3] = s[i + 3];
        }
        else if (miniBlockSize <= 1) {
            t[i] = s[i];
        }
        else {
            printf("something wrong!");
        }
    }
}

/*
copy a number of blocks to target positions (cuda version)
>> source - data array (head of the blocks) to copy from
>> blockSize - size of block
>> blockNum - number of blocks
>> target - target data array
>> targetBlocks - target positions of the copy (on the device)
>> myMem - memory pool
*/
void CudaCopyBlocks(void * source, int blockSize, int blockNum, void * target, int * targetBlocks, XMem * myMem)
{
    CheckNTErrors((myMem != NULL), "No memory pool!");
    CheckNTErrors((myMem->devID >= 0), "Wrong device to run!");
    CheckNTErrors((blockSize % sizeof(DTYPE) == 0), "Unsupported block size!");

    int cudaGrids[3];
    int cudaBlocks[3];
    int bSize = blockSize / sizeof(DTYPE);

    if (bSize % 4 == 0) {
        GDevs.GetCudaThread2D(myMem->devID, bSize / 4, blockNum, MAX_INT, cudaGrids, cudaBlocks);
        KernelCopyBlocks<4> << <dim3(cudaGrids[0], cudaGrids[1]), dim3(cudaBlocks[0], cudaBlocks[1]) >> >
            ((DTYPE*)source, bSize, blockNum, (DTYPE*)target, targetBlocks);
    }
    else {
        GDevs.GetCudaThread2D(myMem->devID, bSize, blockNum, MAX_INT, cudaGrids, cudaBlocks);
        KernelCopyBlocks<1> << <dim3(cudaGrids[0], cudaGrids[1]), dim3(cudaBlocks[0], cudaBlocks[1]) >> >
            ((DTYPE*)source, bSize, blockNum, (DTYPE*)target, targetBlocks);
    }
}
#endif // USE_CUDA

} // namespace nts(NiuTrans.Tensor)