/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Lin Ye (email: linye2015@outlook.com) 2018-06-20
*/

#include "../XTensor.h"
#include "../XDevice.h"
#include "../core/Normalize.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1: normalized the data with normal distribution 
* In this case, dim=0.
*/
bool TestNormalize1()
{
	/* a source tensor of size 2 * 3 */
	int sOrder = 2;
	int * sDimSize = new int[sOrder];
	sDimSize[0] = 2;
	sDimSize[1] = 3;

	int sUnitNum = 1;
	for (int i = 0; i < sOrder; i++)
		sUnitNum *= sDimSize[i];

	/* a target tensor of size 2 * 3 */
	int tOrder = 2;
	int * tDimSize = new int[tOrder];
	tDimSize[0] = 2;
	tDimSize[1] = 3;

	int tUnitNum = 1;
	for (int i = 0; i < tOrder; i++)
		tUnitNum *= tDimSize[i];

	/* a mean tensor of size 3 */
	int meanOrder = 1;
	int * meanDimSize = new int[meanOrder];
	meanDimSize[0] = 3;

	int meanUnitNum = 1;
	for (int i = 0; i < meanOrder; i++)
		meanUnitNum *= meanDimSize[i];

	/* a var tensor of size 3 */
	int varOrder = 1;
	int * varDimSize = new int[varOrder];
	varDimSize[0] = 3;

	int varUnitNum = 1;
	for (int i = 0; i < varOrder; i++)
		varUnitNum *= varDimSize[i];

	/* a a tensor of size 2 * 3 */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 2;
	aDimSize[1] = 3;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	/* a b tensor of size 2 * 3 */
	int bOrder = 2;
	int * bDimSize = new int[bOrder];
	bDimSize[0] = 2;
	bDimSize[1] = 3;

	int bUnitNum = 1;
	for (int i = 0; i < bOrder; i++)
		bUnitNum *= bDimSize[i];

	DTYPE sData[2][3] = { {0.5, -1.0, 2.0},
	                      {3.5, -4.5, 1.0} };
	DTYPE meanData[3] = {2.0, -2.75, 1.5};
	DTYPE varData[3] = {4.5, 6.125, 0.5};
	DTYPE aData[2][3] = { {0.0, 0.0, 0.0},
	                      {0.0, 0.0, 0.0} };
	DTYPE bData[2][3] = { {0.0, 0.0, 0.0},
	                      {0.0, 0.0, 0.0} };
	DTYPE answer[2][3] = { {0.0, 0.0, 0.0},
	                       {0.0, 0.0, 0.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * s = NewTensor(sOrder, sDimSize);
	XTensor * mean = NewTensor(meanOrder, meanDimSize);
	XTensor * var = NewTensor(varOrder, varDimSize);
	XTensor * a = NewTensor(aOrder, aDimSize);
	XTensor * b = NewTensor(bOrder, bDimSize);
	XTensor * t = NewTensor(tOrder, tDimSize);

	/* initialize variables */
	s->SetData(sData, sUnitNum);
	mean->SetData(meanData, meanUnitNum);
	var->SetData(varData, varUnitNum);
	a->SetData(aData, aUnitNum);
	b->SetData(bData, bUnitNum);
	t->SetZeroAll();

	/* call normalize function */
	Normalize(s, t, 0, mean, var, a, b, 0.0);

	/* check results */
	cpuTest = t->CheckData(answer, tUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensors */
	XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
	XTensor * meanGPU = NewTensor(meanOrder, meanDimSize, X_FLOAT, 1.0F, 0);
	XTensor * varGPU = NewTensor(varOrder, varDimSize, X_FLOAT, 1.0F, 0);
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * bGPU = NewTensor(bOrder, bDimSize, X_FLOAT, 1.0F, 0);
	XTensor * tGPU = NewTensor(tOrder, tDimSize, X_FLOAT, 1.0F, 0);

	/* initialize variables */
	sGPU->SetData(sData, sUnitNum);
	meanGPU->SetData(meanData, meanUnitNum);
	varGPU->SetData(varData, varUnitNum);
	aGPU->SetData(aData, aUnitNum);
	bGPU->SetData(bData, bUnitNum);
	tGPU->SetZeroAll();

	/* call normalize function */
	Normalize(sGPU, tGPU, 0, meanGPU, varGPU, aGPU, bGPU, 0.0);

	/* check results */
	gpuTest = tGPU->CheckData(answer, tUnitNum);

	/* destroy variables */
	delete s, t, mean, var, a, b, sGPU, tGPU, meanGPU, varGPU, aGPU, bGPU;
	delete[] sDimSize, tDimSize, meanDimSize, varDimSize, aDimSize, bDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete s, t, mean, var, a, b;
	delete[] sDimSize, tDimSize, meanDimSize, varDimSize, aDimSize, bDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
TODO!!
*/

/* test for Normalize Function */
extern "C"
bool TestNormalize()
{
	XPRINT(0, stdout, "[TEST NORMALIZE] -------------\n");
	bool returnFlag = true, caseFlag = true;

	/* case 1 test */
	caseFlag = TestNormalize1();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 1 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 1 passed!\n");

	/* other cases test */
	/*
	TODO!!
	*/

	if (returnFlag) {
		XPRINT(0, stdout, ">> All Passed!\n");
	}
	else
		XPRINT(0, stdout, ">> Failed!\n");

	XPRINT(0, stdout, "\n");

	return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
