/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Lin Ye (email: linye2015@outlook.com) 2018-06-20
*/


#include "../XTensor.h"
#include "../XDevice.h"
#include "../function/HardTanH.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1: hard tanh function */
bool TestHardTanH1()
{
	/* a x tensor of size 2 * 3 */
	int xOrder = 2;
	int * xDimSize = new int[xOrder];
	xDimSize[0] = 2;
	xDimSize[1] = 3;

	int xUnitNum = 1;
	for (int i = 0; i < xOrder; i++)
		xUnitNum *= xDimSize[i];

	/* a y tensor of size 2 * 3 */
	int yOrder = 2;
	int * yDimSize = new int[yOrder];
	yDimSize[0] = 2;
	yDimSize[1] = 3;

	int yUnitNum = 1;
	for (int i = 0; i < yOrder; i++)
		yUnitNum *= yDimSize[i];

	DTYPE xData[2][3] = { {0.5, -1.0, 2.0},
	                      {3.5, -4.5, 1.0} };
	DTYPE answer[2][3] = { {0.5, -1.0, 1.0},
	                       {1.0, -1.0, 1.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * x = NewTensor(xOrder, xDimSize);
	XTensor * y = NewTensor(yOrder, yDimSize);

	/* initialize variables */
	x->SetData(xData, xUnitNum);
	y->SetZeroAll();

	/* call hardtanh function */
	HardTanH(x, y);

	/* check results */
	cpuTest = y->CheckData(answer, yUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * xGPU = NewTensor(xOrder, xDimSize, X_FLOAT, 1.0F, 0);
	XTensor * yGPU = NewTensor(yOrder, yDimSize, X_FLOAT, 1.0F, 0);

	/* Initialize variables */
	xGPU->SetData(xData, xUnitNum);
	yGPU->SetZeroAll();

	/* call hardtanh function */
	HardTanH(xGPU, yGPU);

	/* check results */
	gpuTest = yGPU->CheckData(answer, yUnitNum);

	/* destroy variables */
	delete x, y, xGPU, yGPU;
	delete[] xDimSize, yDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete x, y;
	delete[] xDimSize, yDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 2: backward computation 
* In this case, lossName=CROSSENTROPY.
*/
bool TestHardTanH2()
{
	/* a x tensor of size 2 * 3 */
	int xOrder = 2;
	int * xDimSize = new int[xOrder];
	xDimSize[0] = 2;
	xDimSize[1] = 3;

	int xUnitNum = 1;
	for (int i = 0; i < xOrder; i++)
		xUnitNum *= xDimSize[i];

	/* a y tensor of size 2 * 3 */
	int yOrder = 2;
	int * yDimSize = new int[yOrder];
	yDimSize[0] = 2;
	yDimSize[1] = 3;

	int yUnitNum = 1;
	for (int i = 0; i < yOrder; i++)
		yUnitNum *= yDimSize[i];

	/* a gold tensor of size 2 * 3 */
	int goldOrder = 2;
	int * goldDimSize = new int[goldOrder];
	goldDimSize[0] = 2;
	goldDimSize[1] = 3;

	int goldUnitNum = 1;
	for (int i = 0; i < goldOrder; i++)
		goldUnitNum *= goldDimSize[i];

	/* a dedy tensor of size 2 * 3 */
	int dedyOrder = 2;
	int * dedyDimSize = new int[dedyOrder];
	dedyDimSize[0] = 2;
	dedyDimSize[1] = 3;

	int dedyUnitNum = 1;
	for (int i = 0; i < dedyOrder; i++)
		dedyUnitNum *= dedyDimSize[i];

	/* a dedx tensor of size 2 * 3 */
	int dedxOrder = 2;
	int * dedxDimSize = new int[dedxOrder];
	dedxDimSize[0] = 2;
	dedxDimSize[1] = 3;

	int dedxUnitNum = 1;
	for (int i = 0; i < dedxOrder; i++)
		dedxUnitNum *= dedxDimSize[i];

	DTYPE xData[2][3] = { {0.5, -1.0, 2.0},
	                      {3.5, -4.5, 1.0} };
	DTYPE yData[2][3] = { {0.5, -1.0, 1.0},
	                       {1.0, -1.0, 1.0} };
	DTYPE goldData[2][3] = { {1.0, 1.0, 1.0},
	                         {1.0, 1.0, 1.0} };
	DTYPE dedyData[2][3] = { {-2.0, 1.0, -1.0},
	                         {-1.0, 1.0, -1.0} };
	DTYPE answer[2][3] = { {-2.0, 1.0, 0.0},
	                       {0.0, 0.0, -1.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * x = NewTensor(xOrder, xDimSize);
	XTensor * y = NewTensor(yOrder, yDimSize);
	XTensor * gold = NewTensor(goldOrder, goldDimSize);
	XTensor * dedy = NewTensor(dedyOrder, dedyDimSize);
	XTensor * dedx = NewTensor(dedxOrder, dedxDimSize);

	/* initialize variables */
	x->SetData(xData, xUnitNum);
	y->SetData(yData, yUnitNum);
	gold->SetData(goldData, goldUnitNum);
	dedy->SetData(dedyData, dedyUnitNum);
	dedx->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(gold, y, x, dedy, dedx, CROSSENTROPY);

	/* check results */
	cpuTest = dedx->CheckData(answer, dedxUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensors */
	XTensor * xGPU = NewTensor(xOrder, xDimSize, X_FLOAT, 1.0F, 0);
	XTensor * yGPU = NewTensor(yOrder, yDimSize, X_FLOAT, 1.0F, 0);
	XTensor * goldGPU = NewTensor(goldOrder, goldDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedyGPU = NewTensor(dedyOrder, dedyDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedxGPU = NewTensor(dedxOrder, dedxDimSize, X_FLOAT, 1.0F, 0);

	/* initialize variables */
	xGPU->SetData(xData, xUnitNum);
	yGPU->SetData(yData, yUnitNum);
	goldGPU->SetData(goldData, goldUnitNum);
	dedyGPU->SetData(dedyData, dedyUnitNum);
	dedxGPU->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(goldGPU, yGPU, xGPU, dedyGPU, dedxGPU, CROSSENTROPY);

	/* check results */
	gpuTest = dedxGPU->CheckData(answer, dedxUnitNum);

	/* destroy variables */
	delete x, y, dedy, dedx, gold, xGPU, yGPU, dedyGPU, dedxGPU, goldGPU;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete x, y, dedy, dedx, gold;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 3: backward computation
* In this case, lossName=SQUAREDERROR.
*/
bool TestHardTanH3()
{
	/* a x tensor of size 2 * 3 */
	int xOrder = 2;
	int * xDimSize = new int[xOrder];
	xDimSize[0] = 2;
	xDimSize[1] = 3;

	int xUnitNum = 1;
	for (int i = 0; i < xOrder; i++)
		xUnitNum *= xDimSize[i];

	/* a y tensor of size 2 * 3 */
	int yOrder = 2;
	int * yDimSize = new int[yOrder];
	yDimSize[0] = 2;
	yDimSize[1] = 3;

	int yUnitNum = 1;
	for (int i = 0; i < yOrder; i++)
		yUnitNum *= yDimSize[i];

	/* a gold tensor of size 2 * 3 */
	int goldOrder = 2;
	int * goldDimSize = new int[goldOrder];
	goldDimSize[0] = 2;
	goldDimSize[1] = 3;

	int goldUnitNum = 1;
	for (int i = 0; i < goldOrder; i++)
		goldUnitNum *= goldDimSize[i];

	/* a dedy tensor of size 2 * 3 */
	int dedyOrder = 2;
	int * dedyDimSize = new int[dedyOrder];
	dedyDimSize[0] = 2;
	dedyDimSize[1] = 3;

	int dedyUnitNum = 1;
	for (int i = 0; i < dedyOrder; i++)
		dedyUnitNum *= dedyDimSize[i];

	/* a dedx tensor of size 2 * 3 */
	int dedxOrder = 2;
	int * dedxDimSize = new int[dedxOrder];
	dedxDimSize[0] = 2;
	dedxDimSize[1] = 3;

	int dedxUnitNum = 1;
	for (int i = 0; i < dedxOrder; i++)
		dedxUnitNum *= dedxDimSize[i];

	DTYPE xData[2][3] = { {0.5, -1.0, 2.0},
	                      {3.5, -4.5, 1.0} };
	DTYPE yData[2][3] = { {0.5, -1.0, 1.0},
	                      {1.0, -1.0, 1.0} };
	DTYPE goldData[2][3] = { {1.0, 1.0, 1.0},
	                         {1.0, 1.0, 1.0} };
	DTYPE dedyData[2][3] = { {-0.5, -2.0, 0.0 },
	                         {0.0, -2.0, 0.0 } };
	DTYPE answer[2][3] = { {-0.5, -2.0, 0.0},
	                       {0.0, 0.0, 0.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * x = NewTensor(xOrder, xDimSize);
	XTensor * y = NewTensor(yOrder, yDimSize);
	XTensor * gold = NewTensor(goldOrder, goldDimSize);
	XTensor * dedy = NewTensor(dedyOrder, dedyDimSize);
	XTensor * dedx = NewTensor(dedxOrder, dedxDimSize);

	/* initialize variables */
	x->SetData(xData, xUnitNum);
	y->SetData(yData, yUnitNum);
	gold->SetData(goldData, goldUnitNum);
	dedy->SetData(dedyData, dedyUnitNum);
	dedx->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(gold, y, x, dedy, dedx, SQUAREDERROR);

	/* check results */
	cpuTest = dedx->CheckData(answer, dedxUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensors */
	XTensor * xGPU = NewTensor(xOrder, xDimSize, X_FLOAT, 1.0F, 0);
	XTensor * yGPU = NewTensor(yOrder, yDimSize, X_FLOAT, 1.0F, 0);
	XTensor * goldGPU = NewTensor(goldOrder, goldDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedyGPU = NewTensor(dedyOrder, dedyDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedxGPU = NewTensor(dedxOrder, dedxDimSize, X_FLOAT, 1.0F, 0);

	/* initialize variables */
	xGPU->SetData(xData, xUnitNum);
	yGPU->SetData(yData, yUnitNum);
	goldGPU->SetData(goldData, goldUnitNum);
	dedyGPU->SetData(dedyData, dedyUnitNum);
	dedxGPU->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(goldGPU, yGPU, xGPU, dedyGPU, dedxGPU, SQUAREDERROR);

	/* check results */
	gpuTest = dedxGPU->CheckData(answer, dedxUnitNum);

	/* destroy variables */
	delete x, y, dedy, dedx, gold, xGPU, yGPU, dedyGPU, dedxGPU, goldGPU;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete x, y, dedy, dedx, gold;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 4: backward computation
* In this case, lossName=ONEHOTERROR.
*/
bool TestHardTanH4()
{
	/* a x tensor of size 2 * 3 */
	int xOrder = 2;
	int * xDimSize = new int[xOrder];
	xDimSize[0] = 2;
	xDimSize[1] = 3;

	int xUnitNum = 1;
	for (int i = 0; i < xOrder; i++)
		xUnitNum *= xDimSize[i];

	/* a y tensor of size 2 * 3 */
	int yOrder = 2;
	int * yDimSize = new int[yOrder];
	yDimSize[0] = 2;
	yDimSize[1] = 3;

	int yUnitNum = 1;
	for (int i = 0; i < yOrder; i++)
		yUnitNum *= yDimSize[i];

	/* a gold tensor of size 2 * 3 */
	int goldOrder = 2;
	int * goldDimSize = new int[goldOrder];
	goldDimSize[0] = 2;
	goldDimSize[1] = 3;

	int goldUnitNum = 1;
	for (int i = 0; i < goldOrder; i++)
		goldUnitNum *= goldDimSize[i];

	/* a dedy tensor of size 2 * 3 */
	int dedyOrder = 2;
	int * dedyDimSize = new int[dedyOrder];
	dedyDimSize[0] = 2;
	dedyDimSize[1] = 3;

	int dedyUnitNum = 1;
	for (int i = 0; i < dedyOrder; i++)
		dedyUnitNum *= dedyDimSize[i];

	/* a dedx tensor of size 2 * 3 */
	int dedxOrder = 2;
	int * dedxDimSize = new int[dedxOrder];
	dedxDimSize[0] = 2;
	dedxDimSize[1] = 3;

	int dedxUnitNum = 1;
	for (int i = 0; i < dedxOrder; i++)
		dedxUnitNum *= dedxDimSize[i];

	DTYPE xData[2][3] = { {0.5, -1.0, 2.0},
	                      {3.5, -4.5, 1.0} };
	DTYPE yData[2][3] = { {0.5, -1.0, 1.0},
	                      {1.0, -1.0, 1.0} };
	DTYPE goldData[2][3] = { {1.0, 0.0, 1.0},
	                         {0.0, 1.0, 1.0} };
	DTYPE dedyData[2][3] = { {-0.5, 0.0, 0.0},
	                         {0.0, -2.0, 0.0} };
	DTYPE answer[2][3] = { {-0.5, 0.0, 0.0},
	                       {0.0, 0.0, 0.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * x = NewTensor(xOrder, xDimSize);
	XTensor * y = NewTensor(yOrder, yDimSize);
	XTensor * gold = NewTensor(goldOrder, goldDimSize);
	XTensor * dedy = NewTensor(dedyOrder, dedyDimSize);
	XTensor * dedx = NewTensor(dedxOrder, dedxDimSize);

	/* initialize variables */
	x->SetData(xData, xUnitNum);
	y->SetData(yData, yUnitNum);
	gold->SetData(goldData, goldUnitNum);
	dedy->SetData(dedyData, dedyUnitNum);
	dedx->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(gold, y, x, dedy, dedx, ONEHOTERROR);

	/* check results */
	cpuTest = dedx->CheckData(answer, dedxUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensors */
	XTensor * xGPU = NewTensor(xOrder, xDimSize, X_FLOAT, 1.0F, 0);
	XTensor * yGPU = NewTensor(yOrder, yDimSize, X_FLOAT, 1.0F, 0);
	XTensor * goldGPU = NewTensor(goldOrder, goldDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedyGPU = NewTensor(dedyOrder, dedyDimSize, X_FLOAT, 1.0F, 0);
	XTensor * dedxGPU = NewTensor(dedxOrder, dedxDimSize, X_FLOAT, 1.0F, 0);

	/* initialize variables */
	xGPU->SetData(xData, xUnitNum);
	yGPU->SetData(yData, yUnitNum);
	goldGPU->SetData(goldData, goldUnitNum);
	dedyGPU->SetData(dedyData, dedyUnitNum);
	dedxGPU->SetZeroAll();

	/* call hardtanhbackward function */
	HardTanHBackward(goldGPU, yGPU, xGPU, dedyGPU, dedxGPU, ONEHOTERROR);

	/* check results */
	gpuTest = dedxGPU->CheckData(answer, dedxUnitNum);

	/* destroy variables */
	delete x, y, dedy, dedx, gold, xGPU, yGPU, dedyGPU, dedxGPU, goldGPU;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete x, y, dedy, dedx, gold;
	delete[] xDimSize, yDimSize, dedyDimSize, dedxDimSize, goldDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
TODO!!
*/

/* test for HardTanH Function */
extern "C"
bool TestHardTanH()
{
	XPRINT(0, stdout, "[TEST HARDTANH] -------------\n");
	bool returnFlag = true, caseFlag = true;

	/* case 1 test */
	caseFlag = TestHardTanH1();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 1 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 1 passed!\n");

	/* case 2 test */
	caseFlag = TestHardTanH2();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 2 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 2 passed!\n");

	/* case 3 test */
	caseFlag = TestHardTanH3();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 3 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 3 passed!\n");

	/* case 4 test */
	caseFlag = TestHardTanH4();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 4 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 4 passed!\n");

	/* other cases test */
	/*
	TODO!!
	*/

	if (returnFlag) {
		XPRINT(0, stdout, ">> All Passed!\n");
	}
	else
		XPRINT(0, stdout, ">> Failed!\n");

	XPRINT(0, stdout, "\n");

	return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
