/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Xu Chen (email: hello_master1954@163.com) 2018-06-29
*/

#include "../XTensor.h"
#include "../XUtility.h"
#include "TIdentity.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1: test Identity function.
* Identity function: y = x 
*/
bool TestIdentity1()
{
    /* a input tensor of size (2, 3) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 3;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE xData[2][3] = { {0.0, 1.0, 2.0}, 
                          {0.5, 0.7, 1.4} };
    DTYPE answer[2][3] = { {0.0, 1.0, 2.0}, 
                          {0.5, 0.7, 1.4} };

    /* CPU test */
    bool cpuTest = true;

    /* create tensors */
    XTensor * x = NewTensor(sOrder, sDimSize);
    XTensor * y = NewTensor(sOrder, sDimSize);

    /* initialize variables */
    x->SetData(xData, sUnitNum);
    y->SetZeroAll();

    /* call Identity function */
    Identity(x, y);
    
    /* check result */
    cpuTest = y->CheckData(answer, sUnitNum);

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensors */
    XTensor * xGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * yGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);

    /* initialize variables */
    xGPU->SetData(xData, sUnitNum);
    yGPU->SetZeroAll();

    /* call Identity function */
    Identity(xGPU, yGPU);
    
    /* check result */
    gpuTest = yGPU->CheckData(answer, sUnitNum);

    /* destroy variables */
    delete x, y;
    delete xGPU, yGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete x, y;
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/* case 2: test IdentityBackward function.
* IdentityBackward function: dE/dx = dE/dy * dy/dx = dE/dy
*/
bool TestIdentity2()
{
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 1;
    sDimSize[1] = 3;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE xData[1][3] = { {0.0, 1.0, 2.0} };
    DTYPE gData[1][3] = { {0.0, 0.0, 1.0} };
    DTYPE answer[3] = {0.090031, 0.244728, -0.334759};

    /* CPU test */
    bool cpuTest = true;

    /* create tensors */
    XTensor * x = NewTensor(sOrder, sDimSize);
    XTensor * y = NewTensor(sOrder, sDimSize);
    XTensor * g = NewTensor(sOrder, sDimSize);
    XTensor * dedy = NewTensor(sOrder, sDimSize);
    XTensor * dedx = NewTensor(sOrder, sDimSize);

    /* initialize variables */
    x->SetData(xData, sUnitNum);
    g->SetData(gData, sUnitNum);
    y->SetZeroAll();
    dedx->SetZeroAll();
    dedy->SetZeroAll();

    /* call Identity function */
    Identity(x, y);

    /* check result */
    printf("CPU Test:\n");
    printf("Identity Result:");
    DTYPE * checkData = (DTYPE*)y->data;
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", checkData[i]);
    }
    printf("\n");
    
    /* call IdentityBackward function */
    IdentityBackward(g, y, x, dedy, dedx, CROSSENTROPY);
    
    /* check result */
    printf("Computer de/dx:");
    checkData = (DTYPE*)dedx->data;
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", checkData[i]);
    }
    printf("\n");

    printf("Real de/dx:");
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", answer[i]);
    }
    printf("\n");

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

        /* create tensors */
    XTensor * xGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * yGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * gGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * dedyGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * dedxGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);

    /* initialize variables */
    xGPU->SetData(xData, sUnitNum);
    gGPU->SetData(gData, sUnitNum);
    yGPU->SetZeroAll();
    dedxGPU->SetZeroAll();
    dedyGPU->SetZeroAll();

    /* call Identity function */
    Identity(xGPU, yGPU);

    /* check result */
    printf("\nGPU Test:\n");
    printf("Identity Result:");
    checkData = (DTYPE*)y->data;
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", checkData[i]);
    }
    printf("\n");

    /* call IdentityBackward function */
    IdentityBackward(gGPU, yGPU, xGPU, dedyGPU, dedxGPU, CROSSENTROPY);
    
    /* check result */
    printf("Computer de/dx:");
    checkData = (DTYPE*)dedxGPU->data;
    int size = sUnitNum * dedxGPU->unitSize;
    DTYPE * copy = new DTYPE[size];
    XMemCopy(copy, -1, checkData, dedxGPU->devID, size);
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", copy[i]);
    }
    printf("\n");

    printf("Real de/dx:");
    for (int i = 0; i < sUnitNum; i++) {
        printf("\t%f", answer[i]);
    }
    printf("\n");

    /* destroy variables */
    delete x, y, g, dedx, dedy;
    delete xGPU, yGPU, gGPU, dedxGPU, dedyGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete x, y, g, dedx, dedy;
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
    TODO!!
*/

/* test for Identity Function */
extern "C"
bool TestIdentity()
{
    XPRINT(0, stdout, "[TEST Identity] -------------\n");
    bool returnFlag = true, caseFlag = true;

    /* case 1 test */
    caseFlag = TestIdentity1();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 1 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 1 passed!\n");

    /* case 2 test */
    caseFlag = TestIdentity2();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 2 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 2 passed!\n");

    /* other cases test */
    /*
    TODO!!
    */

    if (returnFlag) {
        XPRINT(0, stdout, ">> All Passed!\n");
    }
    else
        XPRINT(0, stdout, ">> Failed!\n");

    XPRINT(0, stdout, "\n");

    return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
