/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Lin Ye (email: linye2015@outlook.com) 2018-06-15
*/

#include "../XTensor.h"
#include "../XDevice.h"
#include "../core/Power.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1: get the power(a, p) 
* In this case, p=2.
*/
bool TestPower1()
{
	/* a tensor of size 3 * 2 */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {1.0, 2.0},
	                      {3.0, 4.0},
	                      {5.0, 6.0} };
	DTYPE answer[3][2] = { {1.0, 4.0},
	                       {9.0, 16.0},
	                       {25.0, 36.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);

	/* initialize variables */
	a->SetData(aData, aUnitNum);

	/* call power function */
	Power(a, 2.0);

	/* check results */
	cpuTest = a->CheckData(answer, aUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);

	/* call power function */
	Power(aGPU, 2.0);

	/* check results */
	gpuTest = aGPU->CheckData(answer, aUnitNum, 0.0001F);

	/* destroy variables */
	delete a, aGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 2: get the power(a, p)
* In this case, p=1.
*/
bool TestPower2()
{
	/* a tensor of size 3 * 2 */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {0.0, 1.0},
	                      {2.0, 3.0},
	                      {4.0, 5.0} };
	DTYPE answer[3][2] = { {0.0, 1.0},
	                       {2.0, 3.0},
	                       {4.0, 5.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);

	/* initialize variables */
	a->SetData(aData, aUnitNum);

	/* call power function */
	Power(a, 1.0);

	/* check results */
	cpuTest = a->CheckData(answer, aUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);

	/* call power function */
	Power(aGPU, 1.0);

	/* check results */
	gpuTest = aGPU->CheckData(answer, aUnitNum);

	/* destroy variables */
	delete a, aGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 3: get the power(a, p)
* In this case, p=0.
*/
bool TestPower3()
{
	/* a tensor of size 3 * 2 */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {0.0, 1.0},
	                      {2.0, 3.0},
	                      {4.0, 5.0} };
	DTYPE answer[3][2] = { {1.0, 1.0},
	                       {1.0, 1.0},
	                       {1.0, 1.0} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);

	/* initialize variables */
	a->SetData(aData, aUnitNum);

	/* call power function */
	Power(a, 0.0);

	/* check results */
	cpuTest = a->CheckData(answer, aUnitNum);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);

	/* call power function */
	Power(aGPU, 0.0);

	/* check results */
	gpuTest = aGPU->CheckData(answer, aUnitNum);

	/* destroy variables */
	delete a, aGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
TODO!!
*/

/* test for Power Function */
extern "C"
bool TestPower()
{
	XPRINT(0, stdout, "[TEST POWER] -------------\n");
	bool returnFlag = true, caseFlag = true;

	/* case 1 test */
	caseFlag = TestPower1();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 1 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 1 passed!\n");

	/* case 2 test */
	caseFlag = TestPower2();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 2 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 2 passed!\n");

	/* case 3 test */
	caseFlag = TestPower3();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 3 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 3 passed!\n");

	/* other cases test */
	/*
	TODO!!
	*/

	if (returnFlag) {
		XPRINT(0, stdout, ">> All Passed!\n");
	}
	else
		XPRINT(0, stdout, ">> Failed!\n");

	XPRINT(0, stdout, "\n");

	return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
