/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Xu Chen (email: hello_master1954@163.com) 2018-06-13
*/

#include "../XTensor.h"
#include "../core/Unsqueeze.h"
#include "../XList.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1: transform a tensor by merging it along with a dimension. 
* In this case, (3, 2) -> (6), whereToUnsqueeze=1, leadingDim=0.
*/
bool TestUnsqueeze1()
{
    /* a source tensor of size (2, 3) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 3;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    /* a target tensor of size (6, ) */
    int tOrder = 3;
    int * tDimSize = new int[tOrder];
    tDimSize[0] = 2;
    tDimSize[1] = 3;
    tDimSize[2] = 2;

    int tUnitNum = 1;
    for (int i = 0; i < tOrder; i++)
        tUnitNum *= tDimSize[i];

    DTYPE sData[2][3] = { {0.0, 1.0, 2.0},
                          {3.0, 4.0, 5.0} };
    DTYPE answer[2][2][3] = { { {0.0, 1.0, 2.0},
                                {3.0, 4.0, 5.0} },
                              { {0.0, 1.0, 2.0},
                                {3.0, 4.0, 5.0} } };
    DTYPE new_answer[2][3][2] = { { {0.0, 0.0}, 
                                    {1.0, 1.0}, 
                                    {2.0, 2.0} },
                                  { {3.0, 3.0}, 
                                    {4.0, 4.0}, 
                                    {5.0, 5.0} } };
    
    /* CPU test */
    bool cpuTest = true;

    /* create tensors */
    XTensor * s = NewTensor(sOrder, sDimSize);
    XTensor * t = NewTensor(tOrder, tDimSize);

    /* initialize variables */
    s->SetData(sData, sUnitNum);
    t->SetZeroAll();

    /* call Unsqueeze function */
    Unsqueeze(s, t, 2, 2);

    /* check results */
    cpuTest = t->CheckData(new_answer, tUnitNum);

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tGPU = NewTensor(tOrder, tDimSize, X_FLOAT, 1.0F, 0);

    /* Initialize variables */
    sGPU->SetData(sData, sUnitNum);
    tGPU->SetZeroAll();

    /* call Unsqueeze function */
    Unsqueeze(sGPU, tGPU, 2, 2);

    /* check results */
    gpuTest = tGPU->CheckData(new_answer, tUnitNum);

    /* destroy variables */
    delete s, t, sGPU, tGPU;
    delete[] sDimSize, tDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete s, t;
    delete[] sDimSize, tDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
    TODO!!
*/

/* test for Unsqueeze Function */
extern "C"
bool TestUnsqueeze()
{
    XPRINT(0, stdout, "[TEST Unsqueeze] -------------\n");
    bool returnFlag = true, caseFlag = true;

    /* case 1 test */
    caseFlag = TestUnsqueeze1();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 1 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 1 passed!\n");

    /* other cases test */
    /*
    TODO!!
    */

    if (returnFlag) {
        XPRINT(0, stdout, ">> All Passed!\n");
    }
    else
        XPRINT(0, stdout, ">> Failed!\n");

    XPRINT(0, stdout, "\n");

    return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
