/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: LI Yinqiao (li.yin.qiao.2012@hotmail.com) 2018-04-30
*/

#include "../XTensor.h"
#include "../XDevice.h"
#include "../core/Sum.h"

namespace nts { // namespace nts(NiuTrans.Tensor)
/* case 1 */
bool TestSum1()
{
    /* a tensor of size 2 * 4 */
    int order = 2;
    int * dimSize = new int[order];
    dimSize[0] = 2;
    dimSize[1] = 4;

    int unitNum = 1;
    for (int i = 0; i < order; i++)
        unitNum *= dimSize[i];

    DTYPE aData[2][4] = { {0.0,   1.0,   2.0,   3.0},
                          {4.0,   5.0,   6.0,   7.0} };
    DTYPE bData[2][4] = { {1.0,  -1.0,  -3.0,  -5.0}, 
                          {-7.0, -9.0, -11.0, -13.0} };
    DTYPE answer[2][4] = { {1.0,   0.0,  -1.0,  -2.0},
                           {-3.0, -4.0,  -5.0,  -6.0} };

    /* CPU test */
    bool cpuTest = true;

    /* create tensors */
    XTensor * a = NewTensor(order, dimSize);
    XTensor * b = NewTensor(order, dimSize);

    /* initialize variables */
    a->SetData(aData, unitNum);
    b->SetData(bData, unitNum);

    /* call sum function */
    Sum(a, b);

    /* check results */
    cpuTest = a->CheckData(answer, unitNum);

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(order, dimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(order, dimSize, X_FLOAT, 1.0F, 0);

    /* Initialize variables */
    aGPU->SetData(aData, unitNum);
    bGPU->SetData(bData, unitNum);

    /* call sum function */
    Sum(aGPU, bGPU);

    /* check results */
    gpuTest = aGPU->CheckData(answer, unitNum);

    /* destroy variables */
    delete a, b, aGPU, bGPU;
    delete[] dimSize;
    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete a;
	delete b;
    delete[] dimSize;
    return cpuTest;
#endif // USE_CUDA
}

/* case 2 */
bool TestSum2()
{
    /* a tensor of size 2 * 4 */
    int order = 2;
    int * dimSize = new int[order];
    dimSize[0] = 2;
    dimSize[1] = 4;

    int unitNum = 1;
    for (int i = 0; i < order; i++) {
        unitNum *= dimSize[i];
    }
    DTYPE aData[2][4] = { {0.0,   1.0,   2.0,   3.0},
                          {4.0,   5.0,   6.0,   7.0} };
    DTYPE bData[2][4] = { {1.0,  -1.0,  -3.0,  -5.0}, 
                          {-7.0, -9.0, -11.0, -13.0} };
    DTYPE answer[2][4] = { {0.5,  0.5,   0.5,   0.5},
                           {0.5,  0.5,   0.5,   0.5} };
    float beta = 0.5F;

    /* CPU test */
    bool cpuTest = true;

    /* create tensor */
    XTensor * a = NewTensor(order, dimSize);
    XTensor * b = NewTensor(order, dimSize);
    XTensor * c = NewTensor(order, dimSize);

    /* Initalize variables */
    a->SetData(aData, unitNum);
    b->SetData(bData, unitNum);
    c->SetZeroAll();

    /* call sum function */
    Sum(a, b, c, beta);

    /* check results */
    cpuTest = c->CheckData(answer, unitNum);

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(order, dimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(order, dimSize, X_FLOAT, 1.0F, 0);
    XTensor * cGPU = NewTensor(order, dimSize, X_FLOAT, 1.0F, 0);

    /* Initalize variables */
    aGPU->SetData(aData, unitNum);
    bGPU->SetData(bData, unitNum);
    cGPU->SetZeroAll();

    /* call sum function */
    Sum(aGPU, bGPU, cGPU, beta);

    /* check results */
    gpuTest = cGPU->CheckData(answer, unitNum);

    /* destroy variables */
    delete a, b, c, aGPU, bGPU, cGPU;
    delete[] dimSize;
    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete a;
    delete b;
    delete c;
    delete[] dimSize;
    return cpuTest;
#endif // USE_CUDA
}

/* other cases */
/*
    TODO!!
*/

/* test for Sum Function */
extern "C"
bool TestSum() 
{
    XPRINT(0, stdout, "[TEST SUM] tensor summation c = a + b * beta\n");
    bool returnFlag = true, caseFlag = true;

    /* case 1 test */
    caseFlag = TestSum1();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 1 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 1 passed!\n");

    /* case 2 test */
    caseFlag = TestSum2();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 2 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 2 passed!\n");

    /* other cases test */
    /*
        TODO!!
    */

    if (returnFlag) {
        XPRINT(0, stdout, ">> All Passed!\n");
    }
    else
        XPRINT(0, stdout, ">> Failed!\n");

    XPRINT(0, stdout, "\n");

    return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
