/* NiuTrans.Tensor - an open-source tensor library
 * Copyright (C) 2018, Natural Language Processing Lab, Northestern University. 
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Created by: XIAO Tong (xiaotong@mail.neu.edu.cn) 2018-07-31
 */

#include "T2TFNN.h"
#include "T2TUtility.h"
#include "T2TEmbedding.h"
#include "../../tensor/core/CHeader.h"
#include "../../tensor/function/FHeader.h"

namespace transformer
{

/* constructor */
T2TFNN::T2TFNN()
{
    inSize  = -1;
    outSize = -1;
    hSize   = -1;
}

/* deconstructor */
T2TFNN::~T2TFNN()
{
}

/* 
initialize the model 
>> argc - number of arguments
>> argv - list of pointers to the arguments
>> myDevID - device id
>> myMem - the memory pool
*/
void T2TFNN::InitModel(int argc, const char ** argv, int myDevID, XMem * myMem)
{
    devID = myDevID;
    mem = myMem;
    
    float minmax = 0;

    LoadParamInt(argc, argv, "d", &inSize, DEFAULT_BEDDING_SIZE);
    LoadParamInt(argc, argv, "d", &outSize, DEFAULT_BEDDING_SIZE);
    LoadParamInt(argc, argv, "fnnh", &hSize, DEFAULT_BEDDING_SIZE);
    LoadParamFloat(argc, argv, "fnnminmax", &minmax, 0.08F);

    InitTensor2D(&w1, inSize, hSize, X_FLOAT, devID, mem);
    InitTensor1D(&b1, hSize, X_FLOAT, devID, mem);

    InitTensor2D(&w2, hSize, outSize, X_FLOAT, devID, mem);
    InitTensor1D(&b2, outSize, X_FLOAT, devID, mem);

    w1.SetDataRand(-minmax, minmax);
    b1.SetDataRand(-minmax, minmax);
    w2.SetDataRand(-minmax, minmax);
    b2.SetDataRand(-minmax, minmax);
}

/* 
make the network 
y = max(0, x * w1 + b1) * w2 + b2
>> input - the input tensor
>> return - the output tensor 
*/
XTensor T2TFNN::Make(XTensor &input)
{
    XTensor t1;

    /* t1 = max(0, x * w1 + b1) */
    t1 = Rectify(MMul(input, X_NOTRANS, w1, X_NOTRANS) + b1);

    /* result = t1 * w2 + b2 */
    return MMul(t1, X_NOTRANS, w2, X_NOTRANS) + b2;
}


}
