/* NiuTrans.Tensor - an open-source tensor library
 * Copyright (C) 2018, Natural Language Processing Lab, Northestern University.
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * backward computation for data operation
 * $Created by: Xu Chen (email: hello_master1954@163.com) 2018-12-26
 */

#include "XNoder.h"
#include "XBackwardData.h"
#include "../tensor/XName.h"
#include "../tensor/XUtility.h"
#include "../tensor/core/CHeader.h"
#include "../tensor/core/getandset/SetData.h"

namespace nts{

/* compute dE/dx of a node */
void XDataGrad::MakeGrad(XTensor * node, bool isEfficent)
{
    CheckNTErrors(node->grad != NULL, "No gradient found!");

    XLink &income = node->income;
    int operID = income.typeID;

    if(operID == GETANDSET_CONVERTDATATYPE)
        GradConvertDataType(node, isEfficent);
    else if(operID == GETANDSET_INDEXTOONEHOT)
        GradIndexToOnehot(node, isEfficent);
    else if(operID == GETANDSET_ONEHOTTOINDEX)
        GradOnehotToIndex(node, isEfficent);
    else{
        ShowNTErrors("TODO!");
    }
}

/* indicates whether the node is for a data operation */
bool XDataGrad::IsDataOP(XTensor * node)
{
    XLink &income = node->income;
    return (income.typeID & DATA_BASE) != 0;
}

/* 
gradient computation for convert datatype
for
b = converdatatype(a) 
we have
dE/da = convertdatatype(b)
>> node - the node (c) for backward computation
>> isEfficient - indicates whether the computation is in
                 an efficient manner
*/
void XDataGrad::GradConvertDataType(XTensor * node, bool isEfficent)
{
    XLink &income = node->income;
    CheckNTErrors(income.tailNum > 0, "Wrong input tensor number for ConvertDataType!");

    XTensor * input = income.tails[0];

    XNoder::MakeGrad(input);

	XTensor * tmp = NewTensorBuf(input->grad, input->devID, input->mem);
    _ConvertDataType(node->grad, tmp);
	_SumMe(input->grad, tmp);
	DelTensorBuf(tmp);

	node->visitMark = NODE_FINISHED;
}

/* 
gradient computation for OnehotToIndex
for
b = OnehotToIndex(a) 
we have
dE/da = IndexToOnehot(b)
>> node - the node (c) for backward computation
>> isEfficient - indicates whether the computation is in
                 an efficient manner
*/
void XDataGrad::GradOnehotToIndex(XTensor * node, bool isEfficent)
{
    XLink &income = node->income;
    CheckNTErrors(income.tailNum > 0, "Wrong input tensor number for IndexToOnehot!");

    XTensor * input = income.tails[0];

    XNoder::MakeGrad(input);

	node->visitMark = NODE_FINISHED;
}

/* 
gradient computation for IndexToOnehot
for
b = IndexToOnehot(a) 
we have
dE/da = IndexToOnehot(b)
>> node - the node (c) for backward computation
>> isEfficient - indicates whether the computation is in
                 an efficient manner
*/
void XDataGrad::GradIndexToOnehot(XTensor * node, bool isEfficent)
{
    XLink &income = node->income;
    CheckNTErrors(income.tailNum > 0, "Wrong input tensor number for IndexToOnehot!");

    XTensor * input = income.tails[0];

    XNoder::MakeGrad(input);

	node->visitMark = NODE_FINISHED;
}

} // namespace nts(NiuTrans.Tensor)
