/* NiuTrans.Tensor - an open-source tensor library
 * Copyright (C) 2017, Natural Language Processing Lab, Northestern University. 
 * All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * $Created by: Lin Ye (email: linye2015@outlook.com) 2018-08-03
 * $Update by: Lin Ye (email: linye2015@outlook.com) 2019-07-12 float16/int/int8 added
 */

#include "../XTensor.h"
#include "../core/math/Clip.h"
#include "TClip.h"
#include "../core/getandset/ConvertDataType.h"

namespace nts { // namespace nts(NiuTrans.Tensor)

/*
case 1: test Clip function.
Set every entry to its clip value.
*/
bool TestClip1()
{
	/* a tensor of size (3, 2) */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {1.0F, -2.0F},
						  {0.0F, 4.0F},
						  {5.0F, -6.0F} };
	DTYPE answer[3][2] = { {1.0F, -1.0F},
						   {0.0F, 1.0F},
					   	   {1.0F, -1.0F} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);
	XTensor * b = NewTensor(aOrder, aDimSize);
	XTensor * aMe = NewTensor(aOrder, aDimSize);
	XTensor bUser;

	/* initialize variables */
	a->SetData(aData, aUnitNum);
	aMe->SetData(aData, aUnitNum);

	/* call Clip function */
	_Clip(a, b, -1.0, 1.0);
	_ClipMe(aMe, -1.0, 1.0);
	bUser = Clip(*a, -1.0, 1.0);

	/* check results */
	cpuTest = b->CheckData(answer, aUnitNum, 1e-4F) && 
              aMe->CheckData(answer, aUnitNum, 1e-4F) && 
              bUser.CheckData(answer, aUnitNum, 1e-4F);

#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor bUserGPU;

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);
	aMeGPU->SetData(aData, aUnitNum);

	/* call Clip function */
	_Clip(aGPU, bGPU, -1.0, 1.0);
	_ClipMe(aMeGPU, -1.0, 1.0);
	bUserGPU = Clip(*aGPU, -1.0, 1.0);

	/* check results */
	gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) && 
              aMeGPU->CheckData(answer, aUnitNum, 1e-4F) && 
              bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
	delete aGPU;
	delete bGPU;
	delete aMeGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}


/*
case 2: float16 test Clip function.
Set every entry to its clip value.
*/
bool TestClip2()
{
    /* a tensor of size (3, 2) */
    int aOrder = 2;
    int * aDimSize = new int[aOrder];
    aDimSize[0] = 3;
    aDimSize[1] = 2;

    int aUnitNum = 1;
    for (int i = 0; i < aOrder; i++)
        aUnitNum *= aDimSize[i];

    DTYPE aData[3][2] = { {1.0F, -2.0F},
                          {0.0F, 4.0F},
                          {5.0F, -6.0F} };
    DTYPE answer[3][2] = { {1.0F, -1.0F},
                           {0.0F, 1.0F},
                           {1.0F, -1.0F} };

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

    /* create float16 tensor */
    XTensor aHalfGPU;
    XTensor bHalfGPU;
    XTensor aMeHalfGPU;
    XTensor bUserHalfGPU;

    /* Initialize variables */
    aGPU->SetData(aData, aUnitNum);
    aMeGPU->SetData(aData, aUnitNum);

    /* convert data type from float to float16 */
    aHalfGPU = ConvertDataType(*aGPU, X_FLOAT16);
    aMeHalfGPU = ConvertDataType(*aMeGPU, X_FLOAT16);
    bHalfGPU = ConvertDataType(*bGPU, X_FLOAT16);

    /* call clip function */
    _Clip(&aHalfGPU, &bHalfGPU, -1.0, 1.0);
    _ClipMe(&aMeHalfGPU, -1.0, 1.0);
    bUserHalfGPU = Clip(aHalfGPU, -1.0, 1.0);

    /* convert data type from float16 to float */
    _ConvertDataType(&bHalfGPU, bGPU);
    _ConvertDataType(&aMeHalfGPU, aMeGPU);
    bUserGPU = ConvertDataType(bUserHalfGPU, X_FLOAT);

    /* check results */
    gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              aMeGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

    /* destroy variables */
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
    delete[] aDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] aDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/*
case 3: int32 test Clip function.
Set every entry to its clip value.
*/
bool TestClip3()
{
    /* a tensor of size (3, 2) */
    int aOrder = 2;
    int * aDimSize = new int[aOrder];
    aDimSize[0] = 3;
    aDimSize[1] = 2;

    int aUnitNum = 1;
    for (int i = 0; i < aOrder; i++)
        aUnitNum *= aDimSize[i];

    DTYPE aData[3][2] = { {1.0F, -2.0F},
                          {0.0F, 4.0F},
                          {5.0F, -6.0F} };
    DTYPE answer[3][2] = { {1.0F, -1.0F},
                           {0.0F, 1.0F},
                           {1.0F, -1.0F} };

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

    /* create int32 tensor */
    XTensor aInt32GPU;
    XTensor bInt32GPU;
    XTensor aMeInt32GPU;
    XTensor bUserInt32GPU;

    /* Initialize variables */
    aGPU->SetData(aData, aUnitNum);
    aMeGPU->SetData(aData, aUnitNum);

    /* convert data type from float to int32 */
    aInt32GPU = ConvertDataType(*aGPU, X_INT);
    aMeInt32GPU = ConvertDataType(*aMeGPU, X_INT);
    bInt32GPU = ConvertDataType(*bGPU, X_INT);

    /* call clip function */
    _Clip(&aInt32GPU, &bInt32GPU, -1.0, 1.0);
    _ClipMe(&aMeInt32GPU, -1.0, 1.0);
    bUserInt32GPU = Clip(aInt32GPU, -1.0, 1.0);

    /* convert data type from int32 to float */
    _ConvertDataType(&bInt32GPU, bGPU);
    _ConvertDataType(&aMeInt32GPU, aMeGPU);
    bUserGPU = ConvertDataType(bUserInt32GPU, X_FLOAT);

    /* check results */
    gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              aMeGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

    /* destroy variables */
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
    delete[] aDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] aDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/*
case 4: int8 test Clip function.
Set every entry to its clip value.
*/
bool TestClip4()
{
    /* a tensor of size (3, 2) */
    int aOrder = 2;
    int * aDimSize = new int[aOrder];
    aDimSize[0] = 3;
    aDimSize[1] = 2;

    int aUnitNum = 1;
    for (int i = 0; i < aOrder; i++)
        aUnitNum *= aDimSize[i];

    DTYPE aData[3][2] = { {1.0F, -2.0F},
                          {0.0F, 4.0F},
                          {5.0F, -6.0F} };
    DTYPE answer[3][2] = { {1.0F, -1.0F},
                           {0.0F, 1.0F},
                           {1.0F, -1.0F} };

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

    /* create int8 tensor */
    XTensor aInt8GPU;
    XTensor bInt8GPU;
    XTensor aMeInt8GPU;
    XTensor bUserInt8GPU;

    /* Initialize variables */
    aGPU->SetData(aData, aUnitNum);
    aMeGPU->SetData(aData, aUnitNum);

    /* convert data type from float to int8 */
    aInt8GPU = ConvertDataType(*aGPU, X_INT8);
    aMeInt8GPU = ConvertDataType(*aMeGPU, X_INT8);
    bInt8GPU = ConvertDataType(*bGPU, X_INT8);

    /* call clip function */
    _Clip(&aInt8GPU, &bInt8GPU, -1.0, 1.0);
    _ClipMe(&aMeInt8GPU, -1.0, 1.0);
    bUserInt8GPU = Clip(aInt8GPU, -1.0, 1.0);

    /* convert data type from int8 to float */
    _ConvertDataType(&bInt8GPU, bGPU);
    _ConvertDataType(&aMeInt8GPU, aMeGPU);
    bUserGPU = ConvertDataType(bUserInt8GPU, X_FLOAT);

    /* check results */
    gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              aMeGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

    /* destroy variables */
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
    delete[] aDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] aDimSize;

    return cpuTest;
#endif // USE_CUDA
}


/* other cases */
/*
TODO!!
*/

/* test for Clip Function */
bool TestClip()
{
	XPRINT(0, stdout, "[TEST Clip] set every entry to its clip value \n");
	bool returnFlag = true, caseFlag = true;

	/* case 1 test */
	caseFlag = TestClip1();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 1 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 1 passed!\n");

    /* case 2 test */
    caseFlag = TestClip2();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 2 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 2 passed!\n");

    /* case 3 test */
    caseFlag = TestClip3();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 3 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 3 passed!\n");

    /* case 4 test */
    caseFlag = TestClip4();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 4 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 4 passed!\n");

	/* other cases test */
	/*
	TODO!!
	*/

	if (returnFlag) {
		XPRINT(0, stdout, ">> All Passed!\n");
	}
	else
		XPRINT(0, stdout, ">> Failed!\n");

	XPRINT(0, stdout, "\n");

	return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
