/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Lin Ye (email: linye2015@outlook.com) 2018-06-14
* $Update by: Lin Ye (email: linye2015@outlook.com) 2019-07-12 float16/int/int8 added
*/

#include "TNegate.h"
#include "../core/getandset/ConvertDataType.h"

namespace nts { // namespace nts(NiuTrans.Tensor)

/* case 1: set every entry to its minus value */
bool TestNegate1()
{
	/* a tensor of size (3, 2) */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {1.0F, -2.0F}, 
	                      {-3.0F, 4.0F},
	                      {5.0F, -6.0F} };
	DTYPE answer[3][2] = { {-1.0F, 2.0F},
	                       {3.0F, -4.0F},
	                       {-5.0F, 6.0F} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);
    XTensor * b = NewTensor(aOrder, aDimSize);
	XTensor * aMe = NewTensor(aOrder, aDimSize);
    XTensor bUser;

	/* initialize variables */
	a->SetData(aData, aUnitNum);
	aMe->SetData(aData, aUnitNum);

	/* call Negate function */
	_Negate(a, b);
	_NegateMe(aMe);
    bUser = Negate(*a);

	/* check results */
	cpuTest = b->CheckData(answer, aUnitNum, 1e-4F) && aMe->CheckData(answer, aUnitNum, 1e-4F) && bUser.CheckData(answer, aUnitNum, 1e-4F);
    
#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);
	aMeGPU->SetData(aData, aUnitNum);

	/* call Negate function */
    _Negate(aGPU, bGPU);
	_NegateMe(aMeGPU);
    bUserGPU = Negate(*aGPU);

	/* check results */
	gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) && aMeGPU->CheckData(answer, aUnitNum, 1e-4F) && bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 2: set every entry to its minus value */
bool TestNegate2()
{
	/* a tensor of size (3, 2) */
	int aOrder = 2;
	int * aDimSize = new int[aOrder];
	aDimSize[0] = 3;
	aDimSize[1] = 2;

	int aUnitNum = 1;
	for (int i = 0; i < aOrder; i++)
		aUnitNum *= aDimSize[i];

	DTYPE aData[3][2] = { {0.0F, 0.0F},
	                      {0.0F, 0.0F},
	                      {0.0F, 0.0F} };
	DTYPE answer[3][2] = { {-0.0F, -0.0F},
	                       {-0.0F, -0.0F},
	                       {-0.0F, -0.0F} };

	/* CPU test */
	bool cpuTest = true;

	/* create tensors */
	XTensor * a = NewTensor(aOrder, aDimSize);
    XTensor * b = NewTensor(aOrder, aDimSize);
	XTensor * aMe = NewTensor(aOrder, aDimSize);
    XTensor bUser;

	/* initialize variables */
	a->SetData(aData, aUnitNum);
	aMe->SetData(aData, aUnitNum);

	/* call Negate function */
	_Negate(a, b);
	_NegateMe(aMe);
    bUser = Negate(*a);

	/* check results */
	cpuTest = b->CheckData(answer, aUnitNum, 1e-4F) && aMe->CheckData(answer, aUnitNum, 1e-4F) && bUser.CheckData(answer, aUnitNum, 1e-4F);
    
#ifdef USE_CUDA
	/* GPU test */
	bool gpuTest = true;

	/* create tensor */
	XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
	XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

	/* Initialize variables */
	aGPU->SetData(aData, aUnitNum);
	aMeGPU->SetData(aData, aUnitNum);

	/* call Negate function */
    _Negate(aGPU, bGPU);
	_NegateMe(aMeGPU);
    bUserGPU = Negate(*aGPU);

	/* check results */
	gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) && aMeGPU->CheckData(answer, aUnitNum, 1e-4F) && bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
	delete[] aDimSize;

	return cpuTest && gpuTest;
#else
	/* destroy variables */
	delete a;
	delete b;
	delete aMe;
	delete[] aDimSize;

	return cpuTest;
#endif // USE_CUDA
}

/* case 3: float16 set every entry to its minus value */
bool TestNegate3()
{
    /* a tensor of size (3, 2) */
    int aOrder = 2;
    int * aDimSize = new int[aOrder];
    aDimSize[0] = 3;
    aDimSize[1] = 2;

    int aUnitNum = 1;
    for (int i = 0; i < aOrder; i++)
        aUnitNum *= aDimSize[i];

    DTYPE aData[3][2] = { {1.0F, -2.0F},
                          {-3.0F, 4.0F},
                          {5.0F, -6.0F} };
    DTYPE answer[3][2] = { {-1.0F, 2.0F},
                           {3.0F, -4.0F},
                           {-5.0F, 6.0F} };

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensor */
    XTensor * aGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * bGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor * aMeGPU = NewTensor(aOrder, aDimSize, X_FLOAT, 1.0F, 0);
    XTensor bUserGPU;

    /* create float16 tensor */
    XTensor aHalfGPU;
    XTensor bHalfGPU;
    XTensor aMeHalfGPU;
    XTensor bUserHalfGPU;

    /* Initialize variables */
    aGPU->SetData(aData, aUnitNum);
    aMeGPU->SetData(aData, aUnitNum);

    /* convert data type from float to float16 */
    aHalfGPU = ConvertDataType(*aGPU, X_FLOAT16);
    aMeHalfGPU = ConvertDataType(*aMeGPU, X_FLOAT16);
    bHalfGPU = ConvertDataType(*bGPU, X_FLOAT16);

    /* call negate function */
    _Negate(&aHalfGPU, &bHalfGPU);
    _NegateMe(&aMeHalfGPU);
    bUserHalfGPU = Negate(aHalfGPU);
	
    /* convert data type from float16 to float */
    _ConvertDataType(&bHalfGPU, bGPU);
    _ConvertDataType(&aMeHalfGPU, aMeGPU);
    bUserGPU = ConvertDataType(bUserHalfGPU, X_FLOAT);

    /* check results */
    gpuTest = bGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              aMeGPU->CheckData(answer, aUnitNum, 1e-4F) &&
              bUserGPU.CheckData(answer, aUnitNum, 1e-4F);

    /* destroy variables */
    delete aGPU;
    delete bGPU;
    delete aMeGPU;
    delete[] aDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] aDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/*

/* other cases */
/*
TODO!!
*/

/* test for Negate Function */
bool TestNegate()
{
	XPRINT(0, stdout, "[TEST NEGATE] set every entry to its minus value \n");
	bool returnFlag = true, caseFlag = true;

	/* case 1 test */
	caseFlag = TestNegate1();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 1 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 1 passed!\n");

	/* case 2 test */
	caseFlag = TestNegate2();

	if (!caseFlag) {
		returnFlag = false;
		XPRINT(0, stdout, ">> case 2 failed!\n");
	}
	else
		XPRINT(0, stdout, ">> case 2 passed!\n");

    /* case 3 test */
    caseFlag = TestNegate3();

    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 3 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 3 passed!\n");

	/* other cases test */
	/*
	TODO!!
	*/

	if (returnFlag) {
		XPRINT(0, stdout, ">> All Passed!\n");
	}
	else
		XPRINT(0, stdout, ">> Failed!\n");

	XPRINT(0, stdout, "\n");

	return returnFlag;
}

} // namespace nts(NiuTrans.Tensor)
