/* NiuTrans.Tensor - an open-source tensor library
* Copyright (C) 2017, Natural Language Processing Lab, Northestern University.
* All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*   http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

/*
* $Created by: Xu Chen (email: hello_master1954@163.com) 2018-06-27
* $Update by: Lin Ye (email: linye2015@outlook.com) 2019-07-12 float16/int/int8 added
*/

#include "TScaleAndShift.h"
#include "../core/getandset/ConvertDataType.h"

namespace nts { // namespace nts(NiuTrans.Tensor)

/* 
case 1: scale and shift all tensor entires.
p = p * scale + shift
*/
bool TestScaleAndShift1()
{
    /* a input tensor of size (2, 4) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 4;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE sData[2][4] = { {0.0F, 1.0F, 2.0F, 3.0F},
                          {4.0F, 5.0F, 6.0F, 7.0F} };
    DTYPE answer[2][4] = { {0.5F, 2.5F, 4.5F, 6.5F},
                           {8.5F, 10.5F, 12.5F, 14.5F} };

    DTYPE scaleFactor = 2.0F;
    DTYPE shiftFactor = 0.5F;

    /* CPU test */
    bool cpuTest = true;

    /* create tensors */
    XTensor * s = NewTensor(sOrder, sDimSize);
    XTensor * t = NewTensor(sOrder, sDimSize);
    XTensor * tMe = NewTensor(sOrder, sDimSize);
    XTensor tUser;

    /* initialize variables */
    s->SetData(sData, sUnitNum);
    tMe->SetData(sData, sUnitNum);

    /* call ScaleAndShift function */
    _ScaleAndShift(s, t, scaleFactor, shiftFactor);
    _ScaleAndShiftMe(tMe, scaleFactor, shiftFactor);
    tUser = ScaleAndShift(*s, scaleFactor, shiftFactor);

    /* check results */
    cpuTest = t->CheckData(answer, sUnitNum) && 
        tMe->CheckData(answer, sUnitNum) && tUser.CheckData(answer, sUnitNum);

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensors */
    XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tMeGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor tUserGPU;

    /* initialize variables */
    sGPU->SetData(sData, sUnitNum);
    tMeGPU->SetData(sData, sUnitNum);

    /* call ScaleAndShift function */
    _ScaleAndShift(sGPU, tGPU, scaleFactor, shiftFactor);
    _ScaleAndShiftMe(tMeGPU, scaleFactor, shiftFactor);
    tUserGPU = ScaleAndShift(*sGPU, scaleFactor, shiftFactor);

    /* check results */
    gpuTest = tGPU->CheckData(answer, sUnitNum) && 
        tMeGPU->CheckData(answer, sUnitNum) && tUserGPU.CheckData(answer, sUnitNum);

    /* destroy variables */
    delete s;
    delete t;
    delete tMe;
    delete sGPU;
    delete tGPU;
    delete tMeGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete s;
    delete t;
    delete tMe;
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}


/*
case 2: flaot16 scale and shift all tensor entires.
p = p * scale + shift
*/
bool TestScaleAndShift2()
{
    /* a input tensor of size (2, 4) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 4;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE sData[2][4] = { {0.0F, 1.0F, 2.0F, 3.0F},
                          {4.0F, 5.0F, 6.0F, 7.0F} };
    DTYPE answer[2][4] = { {0.5F, 2.5F, 4.5F, 6.5F},
                           {8.5F, 10.5F, 12.5F, 14.5F} };

    DTYPE scaleFactor = 2.0F;
    DTYPE shiftFactor = 0.5F;

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensors */
    XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tMeGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor tUserGPU;

    /* create float16 tensor */
    XTensor sHalfGPU;
    XTensor tHalfGPU;
    XTensor tMeHalfGPU;
    XTensor tUserHalfGPU;

    /* initialize variables */
    sGPU->SetData(sData, sUnitNum);
    tMeGPU->SetData(sData, sUnitNum);

    /* convert data type from float to float16 */
    sHalfGPU = ConvertDataType(*sGPU, X_FLOAT16);
    tMeHalfGPU = ConvertDataType(*tMeGPU, X_FLOAT16);
    tHalfGPU = ConvertDataType(*tGPU, X_FLOAT16);

    /* call scaleandshift function */
    _ScaleAndShift(&sHalfGPU, &tHalfGPU, scaleFactor, shiftFactor);
    _ScaleAndShiftMe(&tMeHalfGPU, scaleFactor, shiftFactor);
    tUserHalfGPU = ScaleAndShift(sHalfGPU, scaleFactor, shiftFactor);

    /* convert data type from float16 to float */
    _ConvertDataType(&tHalfGPU, tGPU);
    _ConvertDataType(&tMeHalfGPU, tMeGPU);
    tUserGPU = ConvertDataType(tUserHalfGPU, X_FLOAT);

    /* check results */
    gpuTest = tGPU->CheckData(answer, sUnitNum) &&
              tMeGPU->CheckData(answer, sUnitNum) &&
              tUserGPU.CheckData(answer, sUnitNum);

    /* destroy variables */
    delete sGPU;
    delete tGPU;
    delete tMeGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/*
case 3: int32 scale and shift all tensor entires.
p = p * scale + shift
*/
bool TestScaleAndShift3()
{
    /* a input tensor of size (2, 4) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 4;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE sData[2][4] = { {0.0F, 1.0F, 2.0F, 3.0F},
                          {4.0F, 5.0F, 6.0F, 7.0F} };
    DTYPE answer[2][4] = { {1.0F, 3.0F, 5.0F, 7.0F},
                           {9.0F, 11.0F, 13.0F, 15.0F} };

    DTYPE scaleFactor = 2.0F;
    DTYPE shiftFactor = 1.8F;

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensors */
    XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tMeGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor tUserGPU;

    /* create int32 tensor */
    XTensor sInt32GPU;
    XTensor tInt32GPU;
    XTensor tMeInt32GPU;
    XTensor tUserInt32GPU;

    /* initialize variables */
    sGPU->SetData(sData, sUnitNum);
    tMeGPU->SetData(sData, sUnitNum);

    /* convert data type from float to int32 */
    sInt32GPU = ConvertDataType(*sGPU, X_INT);
    tMeInt32GPU = ConvertDataType(*tMeGPU, X_INT);
    tInt32GPU = ConvertDataType(tGPU, X_INT);

    /* call scaleandshift function */
    _ScaleAndShift(&sInt32GPU, &tInt32GPU, scaleFactor, shiftFactor);
    _ScaleAndShiftMe(&tMeInt32GPU, scaleFactor, shiftFactor);
    tUserInt32GPU = ScaleAndShift(sInt32GPU, scaleFactor, shiftFactor);

    /* convert data type from int32 to float */
    _ConvertDataType(&tInt32GPU, tGPU);
    _ConvertDataType(&tMeInt32GPU, tMeGPU);
    tUserGPU = ConvertDataType(tUserInt32GPU, X_FLOAT);

    /* check results */
    gpuTest = tGPU->CheckData(answer, sUnitNum) &&
              tMeGPU->CheckData(answer, sUnitNum) &&
              tUserGPU.CheckData(answer, sUnitNum);

    /* destroy variables */
    delete sGPU;
    delete tGPU;
    delete tMeGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}

/*
case 4: int8 scale and shift all tensor entires.
p = p * scale + shift
*/
bool TestScaleAndShift4()
{
    /* a input tensor of size (2, 4) */
    int sOrder = 2;
    int * sDimSize = new int[sOrder];
    sDimSize[0] = 2;
    sDimSize[1] = 4;

    int sUnitNum = 1;
    for (int i = 0; i < sOrder; i++)
        sUnitNum *= sDimSize[i];

    DTYPE sData[2][4] = { {0.0F, 1.0F, 2.0F, 3.0F},
                          {4.0F, 5.0F, 6.0F, 7.0F} };
    DTYPE answer[2][4] = { {1.0F, 3.0F, 5.0F, 7.0F},
                           {9.0F, 11.0F, 13.0F, 15.0F} };

    DTYPE scaleFactor = 2.0F;
    DTYPE shiftFactor = 1.8F;

    /* CPU test */
    bool cpuTest = true;

#ifdef USE_CUDA
    /* GPU test */
    bool gpuTest = true;

    /* create tensors */
    XTensor * sGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor * tMeGPU = NewTensor(sOrder, sDimSize, X_FLOAT, 1.0F, 0);
    XTensor tUserGPU;

    /* create int8 tensor */
    XTensor sInt8GPU;
    XTensor tInt8GPU;
    XTensor tMeInt8GPU;
    XTensor tUserInt8GPU;

    /* initialize variables */
    sGPU->SetData(sData, sUnitNum);
    tMeGPU->SetData(sData, sUnitNum);

    /* convert data type from float to int8 */
    sInt8GPU = ConvertDataType(*sGPU, X_INT8);
    tMeInt8GPU = ConvertDataType(*tMeGPU, X_INT8);
    tInt8GPU = ConvertDataType(*tGPU, X_INT8);

    /* call scaleandshift function */
    _ScaleAndShift(&sInt8GPU, &tInt8GPU, scaleFactor, shiftFactor);
    _ScaleAndShiftMe(&tMeInt8GPU, scaleFactor, shiftFactor);
    tUserInt8GPU = ScaleAndShift(sInt8GPU, scaleFactor, shiftFactor);

    /* convert data type from int8 to float */
    _ConvertDataType(&tInt8GPU, tGPU);
    _ConvertDataType(&tMeInt8GPU, tMeGPU);
    tUserGPU = ConvertDataType(tUserInt8GPU, X_FLOAT);

    /* check results */
    gpuTest = tGPU->CheckData(answer, sUnitNum) &&
              tMeGPU->CheckData(answer, sUnitNum) &&
              tUserGPU.CheckData(answer, sUnitNum);

    /* destroy variables */
    delete sGPU;
    delete tGPU;
    delete tMeGPU;
    delete[] sDimSize;

    return cpuTest && gpuTest;
#else
    /* destroy variables */
    delete[] sDimSize;

    return cpuTest;
#endif // USE_CUDA
}


/* other cases */
/*
TODO!!
*/

/* test for ScaleAndShift Function */
bool TestScaleAndShift()
{
    XPRINT(0, stdout, "[TEST ScaleAndShift] scale and shift all tensor entires\n");
    bool returnFlag = true, caseFlag = true;

    /* case 1 test */
    caseFlag = TestScaleAndShift1();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 1 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 1 passed!\n");

    /* case 2 test */
    caseFlag = TestScaleAndShift2();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 2 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 2 passed!\n");

    /* case 3 test */
    caseFlag = TestScaleAndShift3();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 3 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 3 passed!\n");

    /* case 4 test */
    caseFlag = TestScaleAndShift4();
    if (!caseFlag) {
        returnFlag = false;
        XPRINT(0, stdout, ">> case 4 failed!\n");
    }
    else
        XPRINT(0, stdout, ">> case 4 passed!\n");

    /* other cases test */
    /*
    TODO!!
    */

    if (returnFlag) {
        XPRINT(0, stdout, ">> All Passed!\n");
    }
    else
        XPRINT(0, stdout, ">> Failed!\n");

    XPRINT(0, stdout, "\n");

    return returnFlag;
    }

} // namespace nts(NiuTrans.Tensor)
